package ch.frankel.blog.hibernate.util;

import static java.sql.Types.INTEGER;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.hibernate.HibernateException;
import org.hibernate.usertype.UserType;

/**
 * Custom mapper from a 8-columns integer to a @link {@link Date}.
 * 
 * @author Nicolas Frnkel
 * @since 13 dc. 2009
 */
public class DateIntegerType implements UserType {

    /** The used date format in database. */
    private static final String DATE_FMT = "yyyyMMdd";

    /**
     * @see org.hibernate.usertype.UserType#assemble(java.io.Serializable,
     *      java.lang.Object)
     */
    @Override
    public Object assemble(Serializable cached, Object owner) throws HibernateException {

        return ((Date) cached).clone();
    }

    /**
     * @see org.hibernate.usertype.UserType#deepCopy(java.lang.Object)
     */
    @Override
    public Object deepCopy(Object value) throws HibernateException {

        return ((Date) value).clone();
    }

    /**
     * @see org.hibernate.usertype.UserType#disassemble(java.lang.Object)
     */
    @Override
    public Serializable disassemble(Object value) throws HibernateException {

        return (Date) ((Date) value).clone();
    }

    /**
     * @see org.hibernate.usertype.UserType#equals(java.lang.Object,
     *      java.lang.Object)
     */
    @Override
    public boolean equals(Object x, Object y) throws HibernateException {

        if (x == null) {

            return y == null;
        }

        return x.equals(y);
    }

    /**
     * @see org.hibernate.usertype.UserType#hashCode(java.lang.Object)
     */
    @Override
    public int hashCode(Object x) throws HibernateException {

        return x == null ? 0 : x.hashCode();
    }

    /**
     * @return <code>true</code> since <code>Date</code> is mutable
     * @see org.hibernate.usertype.UserType#isMutable()
     */
    @Override
    public boolean isMutable() {

        return true;
    }

    /**
     * @see org.hibernate.usertype.UserType#nullSafeGet(java.sql.ResultSet,
     *      java.lang.String[], java.lang.Object)
     */
    @Override
    public Object nullSafeGet(ResultSet rs, String[] names, Object owner) throws HibernateException, SQLException {

        Date result = null;

        if (!rs.wasNull()) {

            Integer integer = rs.getInt(names[0]);

            if (integer != null) {

                try {

                    result = new SimpleDateFormat(DATE_FMT).parse(String.valueOf(integer));

                } catch (ParseException e) {

                    throw new HibernateException(e);
                }
            }
        }

        return result;
    }

    /**
     * @see org.hibernate.usertype.UserType#nullSafeSet(java.sql.PreparedStatement,
     *      java.lang.Object, int)
     */
    @Override
    public void nullSafeSet(PreparedStatement ps, Object value, int index) throws HibernateException, SQLException {

        if (value == null) {

            ps.setNull(index, Types.INTEGER);

        } else {

            Integer integer = Integer.valueOf(new SimpleDateFormat(DATE_FMT).format(value));

            ps.setInt(index, integer);
        }
    }

    /**
     * @see org.hibernate.usertype.UserType#replace(java.lang.Object,
     *      java.lang.Object, java.lang.Object)
     */
    @Override
    public Object replace(Object original, Object target, Object owner) throws HibernateException {

        throw new UnsupportedOperationException("Not coded yet");
    }

    /**
     * @see org.hibernate.usertype.UserType#returnedClass()
     */
    @Override
    public Class<?> returnedClass() {

        return Date.class;
    }

    /**
     * @return @link {@link Types#INTEGER}
     * @see org.hibernate.usertype.UserType#sqlTypes()
     */
    @Override
    public int[] sqlTypes() {

        return new int[] { INTEGER };
    }
}
